using System;
using System.Xml;

using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary.Messages
{
	/// <summary>
	/// This class represents RPC response message defined in the VistALink protocol.
	/// </summary>
	public sealed class RpcResponseMessage : VistALinkMessage
	{
		private string _rawRpcResult;

		/// <summary>
		/// VistALink message type string. 
		/// This constant is made public to recognize message XML.
		/// </summary>
		public const string MessageType = "gov.va.med.foundations.rpc.response";

		/// <summary>
		/// VistALink message XSD schema file name string. 		
		/// This constant is made public to recognize message XML.
		/// </summary>
		public const string XsdName = "rpcResponse.xsd";

		// Constants used in XML message serialization/deserialization
		private const string XMLCONSTS_RESPONSE_TYPE_ATTRIBUTE_NAME = "type"; // this attribute is in spec, but it's of no use
		private const string XMLCONSTS_RESPONSE_TYPE_ATTRIBUTE_VALUE = "string"; // dummy value to stuff into attribute
		private const string XMLCONSTS_RESPONSE_NODE_NAME = "Response";

		/// <summary>
		/// Constructor taking RPC returned value value as a parameter.
		/// </summary>
		/// <param name="rawRpcResult">Value returned by RPC.</param>
		public RpcResponseMessage( string rawRpcResult ) 
			: base( MessageType, XsdName )
		{
			_rawRpcResult = rawRpcResult;
		}

		/// <summary>
		/// Standard VistALink message deserialization constructor. 
		/// Allows recreating message object from XML document.
		/// </summary>
		/// <param name="sourceDoc">Source XML document.</param>
		public RpcResponseMessage( XmlDocument sourceDoc )
			: base( MessageType, XsdName, sourceDoc )
		{
			XmlElement _responseElement = XmlUtility.ParseGetRequiredElementByUniqueTagName( sourceDoc, XMLCONSTS_RESPONSE_NODE_NAME );
			
			XmlUtility.ParseCheckRequiredAttribute( _responseElement, XMLCONSTS_RESPONSE_TYPE_ATTRIBUTE_NAME );  // this attribute is in spec, but it's of no use
			
			if( _responseElement.FirstChild == null || _responseElement.FirstChild.NodeType != XmlNodeType.CDATA )
				throw( new XmlParseException( SR.Exceptions.XmlParseCDataNotFoundWhereExpected( _responseElement.Name ) ) );

			_rawRpcResult = _responseElement.FirstChild.InnerText;
		}

		/// <summary>
		/// Overriden method from the base class. Writes XML message body (transforms object to 
		/// the corresponding VistALink protocol XML message).
		/// </summary>
		/// <param name="writer">Open XmlWriter to write to</param>
		protected override void WriteXmlMessageBody( XmlWriter writer )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			writer.WriteStartElement( XMLCONSTS_RESPONSE_NODE_NAME );
			writer.WriteAttributeString( XMLCONSTS_RESPONSE_TYPE_ATTRIBUTE_NAME, XMLCONSTS_RESPONSE_TYPE_ATTRIBUTE_VALUE );
			writer.WriteCData( _rawRpcResult );
			writer.WriteEndElement();
		}

		/// <summary>
		/// Value returned by the RPC call. It must be a string value. 
		/// Caller must take a responsibility of parsing message to the appropriate format. 
		/// </summary>
		public string RawRpcResult
		{
			get
			{
				return _rawRpcResult;
			}
		}
	}
}
